#
# 標準パッケージ graphics(stats, base) の
# 関数によるグラフ作成
#

# データセットを明示的に読込、パッケージ datasets
data(iris)            # データフレーム    
data(mtcars)          # データフレーム    
data(chickwts)        # データフレーム  
data(PlantGrowth)     # データフレーム
data(VADeaths)        # マトリックス（行列）
data(volcano)     　　# マトリックス（行列）
data(HairEyeColor)　  # テーブル・オブジェクト

# 余白の設定、軸の設定(余白を狭く設定)
par(mar = c(4, 5, 2, 2), mgp = c(2, 0.5, 0))
### mar(下余白, 左余白, 上余白, 右余白)、行単位
### mgp(軸～軸ラベル, 軸～目盛ラベル, 軸～軸線)


# (1) 散布図(Scatter Plot)、plot()  ------------

## (a) iris：がく片の長さと幅の層別散布図
### (a-1) シンプルなコード
plot(iris$Sepal.Width, iris$Sepal.Length,
     col = iris$Species)

### (a-2) 外観のカスタマイズ
plot(
  Sepal.Length ~ Sepal.Width, data = iris,
  col = iris$Species,　  # 内部コード 1,2,3 
  xlab = "がく片の幅",　 # x軸の軸ラベル
  ylab = "がく片の長さ", # y軸の軸ラベル
  las = 1)        # 目盛ラベルを軸と水平に表示

### (a-3) データフレームから抽出
df <- iris[, c("Sepal.Width","Sepal.Length")]
plot(df, col = iris$Species)

## (b) iris：がく片の長さと幅の層別散布図
##     凡例を追加
plot(iris$Sepal.Width, iris$Sepal.Length,
     col = iris$Species,
     pch = 21,            # マーカー指定
     xlim = c(2, 5.5))    # 凡例のスペース確保

legend("topright",        # 凡例を表示する位置
       pch = 21,　　　　　# マーカー指定
       cex = 0.7,         # 凡例の文字サイズ
       col = unique(as.numeric(iris$Species)),  
       legend = levels(iris$Species), 
       title = "Species")

## (c) iris：がく片の長さと幅の層別散布図
##     50% 確率楕円を追加
library(car)  　          # 関数 dataEllipse

plot(iris$Sepal.Width, iris$Sepal.Length, 
     col = iris$Species,　# 因子の内部コード1,2,3
     pch = 21)

dataEllipse(
  iris$Sepal.Width, iris$Sepal.Length, 
  groups = iris$Species,  # グループ化する変数
  level = 0.50,           # 確率水準 (50%)
  col = unique(as.numeric(iris$Species)), 
  label.cex = 0.8,
  add = TRUE) 　　　　　　# 既存のグラフに上書き

## (d) iris：がく片の長さと幅の層別散布図
##     plot()で枠を表示、points()でプロット
plot(x = NA,             # 仮のデータ
     xlim = c(2, 4.5),   # x軸の範囲指定
     ylim = c(4, 8),　　 # y軸の範囲指定
     xlab = "がく片の幅", 
     ylab = "がく片の長さ",
     las = 1)            # 目盛ラベルの表示方向

df <- subset(iris, Species == "setosa")
points(df$Sepal.Width, df$Sepal.Length,
      col = "orange", pch = 16)

df <- subset(iris, Species == "versicolor")
points(df$Sepal.Width, df$Sepal.Length,
       col = "blue", pch = 17)

df <- subset(iris, Species == "virginica")
points(df$Sepal.Width, df$Sepal.Length,
       col = "green", pch = 18)

## (e) iris：3品種のがく片の長さと幅の層別散布図
pch_num  <- c(16, 17, 18)      # 品種のマーカーの形
col_name <- c("orange", "blue", "green") # 品種の色

plot(
  Sepal.Length ~ Sepal.Width, 　 # y軸 ~ x軸
  data = iris,                   # データフレーム
  pch = pch_num[iris$Species],   # 品種ごとの形状
  col = col_name[iris$Species],  # 品種ごとの色
  xlim = c(2, 5),  　　　　　　  # x軸の範囲指定
　las = 1,　　　　　　　　　　　 # 目盛ラベルの方向
  xlab = "がく片の幅", ylab = "がくの長さ")

legend(
  x = "topright",          # 凡例の位置
  bty = "n",               # 凡例の枠線非表示
  cex = 0.7,               # 凡例の文字サイズ
  title.cex = 0.8,         # タイトルの文字サイズ
  legend = levels(iris$Species), # 凡例のテキスト
  pch = pch_num,           # マーカーの形状
  col = col_name,　　　　　# マーカーの色
  title = "品種")  　　　　# 凡例のタイトル

## (f) iris：3品種のがく片の長さと幅の層別散布図
##     coplot()によるマルチパネル（ファセット）
##     代わりにパッケージ lattice の関数を利用
coplot(
  Sepal.Length ~ Sepal.Width | Species, 
  data = iris,
  type = "p",            # プロットする形状
  rows = 1,　　　　　　　# パネルを1列に並べる
  xlab = "がく片の幅",
  ylab = "がく片の長さ")   


# (2) 散布図行列(Scatterplot Matrix)、pairs() --

## (a) mtcars：燃費、馬力、重量の散布図行列
pairs(mtcars[, c("mpg", "hp", "wt")],
      labels = c("燃費","馬力","重量"), 
      cex.labels = 1.5)

## (b) mtcars：燃費、馬力、重量の散布図行列
## 相関係数、ヒストグラムを表示(パネル関数の利用)
panel_cor <- function(
    x, y, digits = 2, prefix = "", cex.cor, ...) {
  par(usr = c(0, 1, 0, 1))
  r <- abs(cor(x, y))
  txt <- format(c(r,0.123456789), digits=digits)[1]
  txt <- paste0(prefix, txt)
  if(missing(cex.cor)) cex.cor <- 0.8/strwidth(txt)
  text(0.5, 0.5, txt, cex = cex.cor * r)
}
panel_hist <- function(x, ...) {　
  usr <- par("usr")
  par(usr = c(usr[1:2], 0, 1.5) )
  hist_out <- hist(x, plot = FALSE)
  breaks <- hist_out$breaks; nB <- length(breaks)
  y <- hist_out$counts; y <- y/max(y)
  rect(xleft = breaks[-nB], ybottom = 0, 
       xright = breaks[-1], ytop = y, 
       col = "cyan", ...)
}

pairs(mtcars[, c("mpg", "hp", "wt")],
      diag.panel = panel_hist,
      upper.panel = panel_cor,
      labels = c("燃費","馬力","重量"))


# (3) 線グラフ(Line Graph)
#     plot()、points()、matplot() --------------

## (a) mtcars：mpg(燃費)とcyl(シリンダー数)の関係
##     cylごとにmpgの平均を計算してプロット
agg_out <- aggregate(
  mpg ~ cyl, data = mtcars, 
  FUN = mean)　        

print(agg_out)

plot(
  mpg ~ cyl, data = agg_out, type = "l",
  xlab = "シリンダー数",
  ylab = "燃費",
  las = 1)        # 目盛ラベルの表示の方向

## (b) mtcars：cyl(シリンダー数)とmpg(燃費)の関係
##     をam(トランスミッション)で層別して表示
##　　 cylとamごとにmpgの平均を計算
agg_out <- aggregate(mpg ~ am + cyl, 
                     data = mtcars, FUN = mean)
print(agg_out)

plot(mpg ~ cyl, data = agg_out, 
     xlim = c(3, 9), ylim = c(15, 30), type = "o",
     col = "orange", pch = 16,
     subset = agg_out$am == 0, 　# automaticで抽出
     xlab = "シリンダー数", ylab = "燃費")

points(mpg ~ cyl, data = agg_out, type = "o",
     col = "blue", pch = 17,   
     subset = agg_out$am ==1)  　# manualで抽出

legend("topright",   
       legend = c("automatic", "manual"),
       title = "トランスミッション",
       col = c("orange", "blue"), pch = c(16, 17),
       cex = 0.9, bty = "n", title.cex = 0.9)     

## (c) ランダムウォーク(時系列データ)の線グラフ
## 　　複数系列を１つのグラフに重ねて表示
t <- 1:50
y1 <- 100 + cumsum(rnorm(50, 0, 5))
y2 <- 100 + cumsum(rnorm(50, 0, 3))
y3 <- 100 + cumsum(rnorm(50, 0, 4))
(mx <- cbind(y1, y2, y3))

matplot(x = t, y = mx, 
        type = "l",　# "p" の指定により散布図
        col = c("orange", "blue", "green"),
        xlab = "時間", ylab = "累積和",
        las = 1)

legend("topleft",
       legend = c("y1", "y2", "y3"),
       col = c("orange", "blue", "green"),
       title = "系列",
       lty = 1:3, lwd = 2, bty = "n",
       title.cex = 1.1, cex = 0.8)


# (4) 散布図の特殊な可視化
#     sunflowerplot() 、symbols() --------------

## (a) iris：ひまわりプロット(Sunflower plot)
##     がく片の幅と長さのひまわりプロットを作成
sunflowerplot(
  iris$Sepal.Width, iris$Sepal.Length,
  xlab = "がく片の幅", 
  ylab = "がく片の長さ",
  main = "Iris データの重複を表示",
  col = "blue",　　　# 青い点をプロット
  seg.col = "red", 　# 重複に対して赤い線分で表示
  las = 1)

## (b) mtcars：シンボルプロット(Symbol plot)
## wt(重量)とmpg(燃費)の散布図にhp(馬力)を表示
symbols(
  mtcars$wt, mtcars$mpg, 
  circles = mtcars$hp,   # hpを円の大きさで表現
  inches = 0.3,          # 円の最大寸法を指定
  fg = "white", bg = "steelblue",
  xlab = "重量", ylab = "燃費",
  las = 1,
  main = "車の重量・燃費・馬力の関係")

text(x = mtcars$wt, y = mtcars$mpg, pos = 3, 
     rownames(mtcars), 
     cex = 0.6)　　　　　# 車種名を表示

text(x = 4, y = 35,
     labels = "円のサイズは馬力を表す")


# (5) 箱ひげ図(Box Plot)、boxplot()、plot() ----

## (a) iris：品種別のがく片の長さの分布
### (a-1) plot() 、縦向き、品種別
plot(Sepal.Length ~ Species, data = iris,
     cex.axis = 0.7)     

### (a-2) boxplot()、縦向き、品種別
boxplot(Sepal.Length ~ Species, data = iris,
        cex.axis = 0.7)

### (a-3) boxplot()、横向き、品種別
boxplot(Sepal.Length ~ Species, data = iris, 
        horizontal = TRUE, cex.axis = 0.7) 

### (a-4) boxplot()、縦向き、３品種込み
boxplot(iris$Sepal.Length)     

### (a-5) boxplot()、横向き、３品種込み
boxplot(iris$Sepal.Length)            

## (b) chickwts：飼料の種類が鶏の体重に及ぼす効果
current_par <-
  par(no.readonly = TRUE) 　# パラメータの保存
par(mar = c(4, 6, 1, 1))　　# 余白の設定

boxplot(
  weight ~ feed, data = chickwts, 
  las = 1,
  cex.axis = 0.8,     # 目盛ラベルの文字サイズ
  cex = 1.2,          # 外れ値のマーカーサイズ
  col = "lightgray",　# 箱の背景色
  border = "black",   # 箱枠の色
  coef = 1.5,         # 外れ値を検出する係数
  varwidth = TRUE,    # 箱の幅が n の平方根に比例
  notch = FALSE,      # ノッチの追加なし
  horizontal = TRUE,  # 横向きのグラフ
  xlab = "体重", ylab = "")  

mtext("飼料", side = 2, line = 3.5) # y軸の軸ラベル

par(current_par)   　 #パラメータの復元


# (6) １次元散布図(Strip Chart)、stripchart() --

## (a) ３試験区での観測値の分布(サンプルサイズ小)
group <- rep(c("A", "B", "C"), each = 3)
obs <- c(5, 7, 8, 6, 9, 10, 8, 10, 11)

### (a-1) 横向きのグラフ(vertical=FALSE 規定値)
stripchart(obs ~ group, las = 1)

### (a-2) 縦向きのグラフ
stripchart(obs ~ group, vertical = TRUE)

## (b) PlantGrowth：3試験区の生長量の分布
stripchart(
  weight ~ group, data = PlantGrowth,
  vertical = TRUE,   # 縦方向のグラフ
  method = "overplot", # 点をバラつかせる
  pch = 16, 　　　　 # マーカーの種類
  col = c("orange","green","blue"),
  xaxt = "n",　　　　# x軸の目盛ラベルを非表示
  las = 1,           # 目盛ラベルの方向
  xlab = "試験区",
  ylab = "生長量")

axis(side = 1, at = 1:3, 
     labels = c("対照", "処理1", "処理2"))

## (C) chickwts：飼料の種類が鶏の体重に及ぼす効果
stripchart(
  weight ~ feed, data = chickwts, 
  las = 2,　　　　　　# 目盛ラベル：xは垂直,yは水平
  method = "jitter",  # 点をバラつかせる
  jitter = 0.1,       # ジッターの量 (0〜1)
  pch = 1,            # マーカーの種類
  cex.axis = 0.8,     # 目盛ラベルの文字サイズ
  vertical = TRUE)　　# 縦方向のグラフ

boxplot(
  weight ~ feed, data = chickwts,
  add = TRUE,         # 既存のプロットに追加
  outline = FALSE,　　# 外れ値を非表示
  col = NA,           # 箱の塗りつぶしなし
  border = "red",　   # 箱の枠線の色
  boxwex = 0.8,　　　 # 箱の相対的な大きさを指定
  axes = FALSE)       # 軸を非表示


# (7) ヒストグラム(Histogram)、hist() ----------

## (a) iris："setosa" のがく片の長さのヒストグラム
v <- iris$Sepal.Length[iris$Species == "setosa"]

### (a-1) 既定値(引数 breaks = "Sturges")
hist(v, xlab = "がく片の長さ", ylab = "度数")  

### (a-2) 引数 breaks に区間の概数を渡す
hist(v, breaks = 8)               

### (a-3) 引数 breaks に等間隔の境界値を渡す
hist(v, breaks = seq(4, 6.5, by = 0.25)) 

### (a-4) 引数 breaks に不等間隔の境界値を渡す
hist(v, breaks = c(3, 4.5, 5, 5.5, 6, 6.5, 8.5))

### (a-5) 引数 breaks に計算方式を渡す
hist(v, breaks = "Scott")         

## (b) iris："setosa" のがく片の長さのヒストグラム
##     ヒストグラムに正規分布、密度曲線を追加
y <- iris$Sepal.Length[iris$Species == "setosa"]

hist(y, 
     freq = FALSE,        # 縦軸を確率密度に指定
     col = "lightblue",
     border = "white",
     xlim = c(4, 6),      # x 軸の範囲
     main = "",　　　　　 # タイトルの削除 
     xlab = "がく片の長さ")

m <- mean(y); s <- sd(y)  # 平均と標準偏差の計算
curve(expr = dnorm(x, mean = m, sd = s), 
      add = TRUE,         # 正規分布の曲線を上書
      col = "red", lwd = 1.5)

lines(density(y),         # カーネル密度推定(KDE)
      col = "blue", lwd = 1.5)

## (c) iris：３品種別のがく片の長さのヒストグラム
##     品種ごとに別パネルで描画(マルチパネル)
par(mfrow = c(3, 1))  # 作画領域を 3行× 1列に分割

sp_names <- unique(iris$Species)    # 品種名を取得
x1 <- 3.5; x2 <- 8.5; y1 <- 0; y2 <- 25 # 軸の範囲

for (s in sp_names) {
  y <-subset(iris, Species == s)$Sepal.Length
  m <- mean(y)   
  
  hist(y, xlim = c(x1, x2), ylim = c(y1, y2),
       main = paste("品種：", s),
       xlab = "がく片の長さ", ylab = "度数",
       col = "lightblue", border = "black",
       breaks = seq(x1, x2, by = 0.5))

  abline(v = m, col = "red")
  text(x = m + 0.8, y = par("usr")[4] * 0.8, 
       paste("平均", round(m, 2)),col = "red")
}
par(mfrow = c(1, 1))  # 作業領域をリセット


# (8) 幹葉図(Stem-and-Leaf Plots)、stem() ------

## iris："setosa" のがく片の長さの分布
y <- iris$Sepal.Length[iris$Species == "setosa"]
sort(y)             # v を昇順にソートして出力

stem(y)  　　　     # scale = 1 (既定値)
stem(y, scale = 2)  # scale = 1 の長さの２倍


# (9) 棒グラフ(Bar Graph) 、barplot()、plot() --

## (a) １つの因子ベクトルの集計結果
ans <- factor(
  c("yes","no","yes","yes","no","no","no","no"),
  levels = c("no","yes"))　　

plot(ans)　　　 # levels の指定の順序で並ぶ

## (b) １つの数値ベクトル（名前付き）
y <- c(1.5, 3.8, 4.1, 5.7)
names(y) <- c("A1","A2","A3","A4")

barplot(y)　　

## (c) 数値ベクトル～因子ベクトル(各要素に対応あり)
y <- c(1.5, 3.8, 4.1, 5.7)
group <- factor(c("A1","A2","A3","A4"))

barplot(y ~ group)

## (d) 数値ベクトル～因子ベクトル+因子ベクトル
group <- factor(c("A","A","B","B","C","C"),
              levels = c("B", "C", "A"))
ans <- factor(c("yes","no","yes","no","yes","no"),
              levels = c("yes", "no"))
y <- c(11, 9, 15, 4, 19, 3)
data.frame(group, ans, y)    # 対応関係を表示　

### (d-1) 積上げ棒グラフ(既定値)
barplot(y ~ ans + group,      
        legend.text = TRUE,  # 凡例を表示
        ylim = c(0,40))      # 凡例のスペース確保

### (d-2) 並列棒グラフ
barplot(y ~ ans + group, 
        beside = TRUE,　　　 # 並列棒グラフを指定 
        space = c(0.1, 0.5), # 棒と棒の間隔
        legend.text = TRUE)　# 凡例を表示

## (e) 複数の数値ベクトル～因子ベクトル
group <- factor(c("A","B","C"))
y1 <- c( 1, 12,  9)
y2 <- c(15,  5, 11)
y3 <- c( 9,  8,  7)
mx <- cbind(y1, y2, y3) # マトリックス作成
print(mx)

### (e-1) 積上げ棒グラフ(既定値)
barplot(mx ~ group,
        legend.text = TRUE,
        args.legend = list(horiz = TRUE),
        ylim = c(0, 40))

### (e-2) 並列棒グラフ
barplot(mx ~ group, 
        beside = TRUE,      # 並列棒グラフ
        legend.text = TRUE,
        args.legend = list(horiz = TRUE),
        ylim = c(0, 25))

## (f) マトリックス、%表示の積上げ棒グラフ
mx <- matrix(c(10, 12, 18, 21,
               15, 18,  5, 10,
                8, 11, 10,  3),
             nrow = 3, ncol = 4, byrow = TRUE)
dimnames(mx) <- list(
  ans = c("－", "＋", "＋＋"),       # 行の項目名
  group = c("G1", "G2", "G3","G4"))  # 列の項目名
print(mx)

### (f-1)　積上げ棒グラフ(実数)
barplot(mx,
        ylim = c(0, 50), las = 1,
        legend.text = TRUE,　# 凡例を表示
        args.legend = list(　# 凡例のカスタマイズ
          x = "top", 
          horiz = TRUE, 
          bty = "n"),
        beside = FALSE)　    # 積上げ棒グラフ

### (f-2) 積上げ棒グラフ(割合)
mx_percent <- prop.table(mx, margin = 2) * 100
barplot(mx_percent, 　　　　 # % データ
        ylim = c(0, 120), 
        legend.text = TRUE,  # 凡例を表示
        args.legend = list(　# 凡例のカスタマイズ
          x = "top", 
          horiz = TRUE,
          bty = "n"),
        beside = FALSE)　　  # 積上げ棒グラフ

### (f-3) 並列棒グラフ
barplot(mx,
        ylim = c(0, 30),
        las = 1,           　# 目盛ラベルの方向
        legend.text = TRUE,  # 凡例を表示
        args.legend = list(  # 凡例のカスタマイズ
          x = "top",
          horiz = TRUE, 　 
          bty = "n",
          title = "副反応"),
        beside = TRUE)　　   # 並列棒グラフ

## (g) mtcars：集計値(テーブル・オブジェクト)
tb <- table(mtcars$am, mtcars$cyl)
print(tb)

barplot(
  tb,
  ylim = c(0, 17), ylab = "台数",
  las = 1,
  names.arg = c("4気筒", "6気筒", "8気筒"),
  legend.text = c("automatic", "manual"),
  args.legend = list(        # 凡例の細部設定
    x = "topleft",           # 左上に配置
    bty = "n",        　　　 # 枠なし
    title = "transmission",  # 凡例のタイトル
    ncol = 1,                # 1 列
    cex = 0.8,               # 文字サイズ
    inset = 0.02             # 内側に少し余白
  )
)


# (10) ドットプロット(Cleveland Dot Plot)
#       dotchart() -----------------------------

## (a) VADeaths：死亡率を地域・性別と年齢層で比較
print(VADeaths)  　　　　　 # マトリックスを表示

dotchart(VADeaths,
         main = "年齢層・グループ別死亡率",
         xlab = "死亡率 (‰)",
         color = "blue",  　# 点の色
         gcolor = "red",　　# グループラベルの色
         cex = 0.8,     　　# 文字サイズ
         pch = 19)      　　# 点のシンボル

## (b) mtcars：データフレーム
df <- mtcars[order(mtcars$mpg), ]  # 昇順に並び替え
g <- factor(df$cyl,                # 数値→因子
            labels = c("4気筒", "6気筒", "8気筒"))
colors <- c("red", "blue", "green")[as.numeric(g)]

data.frame(df$mpg, g, colors)

dotchart(
  df$mpg,
  groups = g,       # グループ分けする因子ベクトル
  labels = rownames(df),
  main = "気筒数別・車種ごとの燃費",
  xlab = "燃費(MPG)",
  gcolor = "black", # グループ名の色
  color = colors,   # グループごとの色
  pch = 19)


# (11) 円グラフ(Pie Chart)、pie() --------------

## (a) 4水準、反時計回り、開始位置は3時(規定値)
y1 <- c(30, 25, 20, 15)
pie(y1)    # 規定値での描画

## (b) 4水準、時計回りで開始位置は12時（既定値）
y2 <- c(A1 = 30, A2 = 25, A3 = 20, A4 = 15)
pie(y2, clockwise = TRUE)　# 省略 init.angle=90

## (c) 4水準、ラベル表示
y3 <- c(30, 25, 20, 15)
lab <- c("A1", "A2", "A3", "A4")
pie(y3, clockwise = TRUE, labels = lab)

## (d) 4水準、% 表示
y4 <- c(30, 25, 20, 15)
lab <- c("A1", "A2", "A3", "A4")
p <- round(y4/sum(y4) * 100, 1)  # 割合を四捨五入
tx <- paste(lab, "\n", p, "%")　 # ラベルの文字列
pie(y4, 
    init.angle = 90, clockwise = TRUE, # 一般的
    labels = tx,
    cex = 1.1)


# (12) モザイク図(Mosaic Plot) 
#      mosaicplot()、plot() --------------------

## (a) 男女別の回答数の比較(対応のある因子ベクトル)
sex <- factor( c(rep("male", 4), rep("female", 5)),
               levels = c("male", "female"))
answer <- factor(
  c("no",  "no", "yes", "yes",
    "no",  "no", "yes",  "no",  "no"),
  levels = c("no", "yes"))
tb_out <- table(sex, answer)  #因子の順序に注意
print(tb_out)

### (a-1) sex を横軸、answer を縦軸に配置
plot(sex, answer)　　　# スパインプロット(参考)
plot(answer ~ sex)     # スパインプロット(参考)
plot(tb_out)　　　　　　# モザイク図

### (a-2) sex を横軸、answer を縦軸に配置
mosaicplot(sex ~ answer)　　 　# plotの設定と逆
mosaicplot(~ sex + answer,cex.axis=1.2)　# 推奨
mosaicplot(tb_out,cex.axis=1.2)

# (b) HairEyeColor： 男性の髪と目の色の関係
mx <- HairEyeColor[, , 1]  # 男性を抽出
print(mx)

mosaicplot(mx,
           main = "",
           xlab = "髪の色 (Hair)",
           ylab = "目の色 (Eye)",
           las = 2,
           color = TRUE)


# (13) スパインプロット(Spine Plot)
#      spineplot()、plot() ---------------------

## (a) 処理群(説明変数)と反応(応答変数)の因果関係
group <- factor( c(rep("ctrl", 4), rep("trt", 5)),
               levels = c("ctrl", "trt"))
answer <- factor(
  c("no",  "no", "yes", "yes",
    "no",  "no", "yes",  "no",  "no"),
  levels = c("no", "yes"))

tb_out <- table(group, answer)  #因子の順序に注意
print(tb_out)

### (a-1) group を横軸、answer を縦軸に配置
plot(group, answer)　　　
plot(answer ~ group) 

### (a-2) group を横軸、answer を縦軸に配置
spineplot(answer ~ group)    
spineplot(tb_out)


# (14) スピノグラム(Spinogram) 、spineplot() ---

## (a) iris：説明変数応答変数の因果関係
spineplot(
  Species ~ Sepal.Length, data = iris,
  xlab = "がく片の長さ (cm)", 
  ylab = "品種",
  col = c("lightblue", "lightgreen", "lightcoral"))


# (15) Cohen-Friendlyの連関図(Association Plot)
#      アソシエーション分析　assocplot() -------

## (a) HairEyeColor：女性の髪の色と目の色の関係
mx15 <- HairEyeColor[,, "Female"] 　# 女性を抽出
print(mx15)

# par(cex.axis = 0.7, cex.lab = 0.7) # 文字を調整
assocplot(mx15, 
          space = 0.3,
          xlab = "髪の色", ylab = "目の色")


# (16)フォーフォールド表示(Fourfold Display)、
#     (４分表示)、fourfoldplot() -------------

## 3 組の 2×2 分割表を比較
mx1 <- matrix(c( 1, 14,
                 8,  2), ncol = 2, byrow = TRUE)
mx2 <- matrix(c( 4, 11,
                 7,  3), ncol = 2, byrow = TRUE)
mx3 <- matrix(c(18, 12,
                11,  9), ncol = 2, byrow = TRUE)
arr <- array(
  c(mx1, mx2, mx3),
  dim = c(2, 2, 3),    # 2行×2列×3個
  dimnames = list(
    group  = c("旧処方", "新処方"), 　　# 行名
    answer = c("有効", "無効"),     　　# 列名
    table  = c("例-1", "例-2", "例-3")  # グループ
  ))

fourfoldplot(arr, 
             conf.level = 0.95, mfcol = c(2, 2),
             color = c("orange", "skyblue"))


# (17) １標本 Q-Q プロット(正規 Q-Q プロット)
#     (Q-Q Plot、Normal Q-Q Plot)、qqnorm() ----

## (a) t分布の乱数と正規分布を比較するQ-Qプロット
y1 <- rt(200, df = 5) # t分布(自由度5)の乱数200個

qqnorm(y1)            # 正規 Q-Q プロット
qqline(y1,
       col = "red")     # 赤い参照線を追加　　　
       
## (b) iris：setosa のがく片の長さの分布と
##     正規分布を比較する Q-Q プロット
y1 <- subset(
  iris, Species == "setosa")$Sepal.Length

qqnorm(y1,
       xlab = "理論的な正規分布の分位点",
       ylab = "サンプルの分位点",
       pch = 1,
       datax = FALSE, 　# データを y軸に割当て
       col = "blue")

qqline(y1,   　　　　　　　　# 参照線を追加　　　
       prob = c(0.25, 0.75), # 参照線の通過点 
       distribution = qnorm, # 理論分布の関数
       col = "red")　　 　　 # ↑ 上記2行は規定値　


# (18) 2標本 Q-Qプロット(Quantile-Quantile Plot)
#      qqplot() --------------------------------

## (a) ２つの正規乱数の分布を比較するQ-Qプロット
y1 <- rnorm(100)
y2 <- rnorm(100)

qqplot(y1, y2)
abline(a = 0, b = 1,　# 参照線、a：切片、b：傾き
       col = "red")

## (b) iris：２品種のがく片の長さの分布を比較する
##            Q-Q プロット  
y1 <- iris$Sepal.Length[iris$Species=="setosa"]
y2 <- iris$Sepal.Length[iris$Species=="virginica"]
var_max <- max(max(y1), max(y2)) # 2変量の最大値
var_min <- min(min(y1), min(y2)) # 2変量の最小値

qqplot(
  y1, y2,
  xlim = c(var_min, var_max), 
  ylim = c(var_min, var_max),
  xlab = "Setosa のがく片の長さ",
  ylab = "virginica のがく片の長さ",
  conf.level = 0.95,
  conf.args = list(
    exact = NULL,              # 既定値
    simulate.p.value = FALSE,  # 既定値
    col = "skyblue", border = "black"))

abline(a = median(y2) - median(y1), 
       b = 1, 
       col = "red", lty = 2)　 # 参照線

## (c) 正規乱数と標準正規乱数を比較するQ-Qブロット
y1 <- rnorm(100)
y2 <- rnorm(100, mean = 2, sd = 3)

qqplot(y1, y2)

q1 <- quantile(y1, probs = c(0.25, 0.75))
q2 <- quantile(y2, probs = c(0.25, 0.75))
slope <- diff(q2) / diff(q1)       # 参照線の傾き
intercept <- q2[1] - slope * q1[1] # 参照線の切片
abline(a = intercept,
       b = slope,    　# 第1四分位数と第3四分位数
       col = "red")　　# を通る参照線

# (19) 等高線プロット(Contour Plot)、contour() -

## (a) volcano：火山の 10m×10m の標高値から作成
head(volcano)　　　　 # データを一部表示

z <- volcano          # 等高線のマトリックス(行列)
x <- 10 * (1:nrow(z)) # 行番号を10 mに変換 (S～N)
y <- 10 * (1:ncol(z)) # 列番号を10 mに変換 (E～W)

contour(x, y, z,      # 位置引数
        asp = 1,      # x軸とy軸を同じスケール
        las = 1)　    # 目盛ラベルを水平に表示

contour(volcano)　　　# マトリックスをそのま描画

## (b) ２次元正規分布の等高線
r  <- 0.5　                        # 母相関係数
xx <- seq(-3, 3, length.out = 50)  # x軸の値 50個 
yy <- seq(-3, 3, length.out = 50)  # y軸の値 50個

### 2次元正規分布の確率密度関数 (PDF) を定義
f_pdf <- function(x, y, rho = 0.5) {
  exponent <- -(x^2 - 2 * rho * x * y + y^2) /
              (2 * (1 - rho^2))
  constant <- 1 / (2 * pi * sqrt(1 - rho^2))
  return(constant * exp(exponent))
}

### z の計算と等高線の描画、z軸の値(50×50 個)
zz <- outer(
  xx, yy, FUN = f_pdf, rho = r)　

contour(xx, yy, zz,
        asp = 1,
        main = paste0("2次元正規分布の等高線 \n
        　　         (rho = ", r,")"))


# (20) ３次元透視図(3D Perspective Diagram)
#     persp() ----------------------------------

## (a) volcano：火山の10m×10mのグリッド上の地形情報
head(volcano)
z <- volcano            # 標高値を付値
x <- 10 * (1:nrow(z))   # 行番号を10 mに補正(S～N)
y <- 10 * (1:ncol(z))   # 列番号を10 mに補正(E～W)

persp(
  x, y, z, 
  theta = 135,      # 横方向の回転角度(方位角)
  phi = 30,         # 縦方向の回転角度(仰角)
  shade = 0.4,      # 陰影の強さ(3D感を出す)
  ticktype = "simple",  # 目盛ラベル非表示  
  border = NA,          # 面を囲む線の色
  box = FALSE,　　　    # 枠線を非表示
  col = "lightgreen")       

## (b) ２次元正規分布の3D曲面プロット
r  <- 0.9          　      　　　　# 母相関係数
xx <- seq(-3, 3, length.out = 50)  # x軸の範囲
yy <- seq(-3, 3, length.out = 50)  # y軸の範囲 

### 2次元正規分布の確率密度関数 (PDF) を定義
f_pdf <- function(x, y, rho = 0.5) {
  z1 <- x; z2 <- y　　# 標準化変数
  exponent <- -(z1^2 - 2 * rho * z1 * z2 + z2^2) /
    　　　    (2 * (1 - rho^2))
  constant <- 1 / (2 * pi * sqrt(1 - rho^2))
  return(constant * exp(exponent))
}

zz <- outer(xx, yy, FUN = f_pdf, rho = r)

persp(
  xx, yy, zz,
  main = paste0("2次元正規分布 \n (rho = ", r,")"),
  theta = -10,           # z軸周りの回転角度(方位角)
  phi = 35,              # x軸周りの回転角度(仰角)
  expand = 1,            # z軸方向の伸縮率
  col = "lightblue",     # 表面の色
  shade = 0.4,           # 陰影の強さ (3D感を出す)
  ticktype = "detailed", # 軸の目盛りを非表示、矢印
  border = NA,           # メッシュの線を消す
  cex.axis = 0.5,        # 目盛ラベルの文字サイズ
  cex.lab = 0.5)         # 軸ラベルの文字サイズ


# (21) ヒートマップ(Heat Map)、image() ---------

## (a) volcano：火山の等高線情報のヒートマップ
head(volcano)

z <- volcano          # 等高線のマトリックス(行列)
x <- 10 * (1:nrow(z)) # 行番号を10 mに変換
y <- 10 * (1:ncol(z)) # 列番号を10 mに変換

image(x, y, z,
      col = terrain.colors(100),
      asp = 1,        # 地形を正しく表示するため 
      las = 1,
      xlab = "X (m)", ylab = "Y (m)")

contour(x, y, z, add = TRUE) # 等高線を追加

## (b) mtcars：相関係数行列をヒートマップで可視化
mx <- cor(mtcars)　         　# 相関係数行列を付値
head(mx)　　　　　　　　　　　# マトリックスを表示
x <- 1:ncol(mx); y <- 1:nrow(mx)

image(
  x, y, mx,
  xlab = "", ylab = "",       # axisで設定
  axes = FALSE, 　　　   　   # 標準の軸を非表示
  col = colorRampPalette(     # グラデーション
    c("blue", "white", "red"))(100),  
  zlim = c(-1, 1))　　　　　　# Z値の範囲指定

axis(side = 1, at = x,        # x軸目盛ラベル表示
     labels = colnames(mx), 　# 車種を表示
     cex.axis = 0.8, las = 2) 
axis(side = 2, at = y,        # y軸目盛ラベル表示
     labels = rownames(mx), 　# 車種を表示
     cex.axis = 0.8, las = 2)                 
text(expand.grid(x, y),       # x,y の全ての組合せ
     labels = round(as.vector(mx), 2), 
     cex = 0.4)       　      # mxをベクトルに変換


# (22) 関数プロット(Function plot)、plot()、curve()

## (a) 正規分布(Normal Distribution)の確率密度曲線
### (a-1) 標準正規分布を描画、関数オブジェクトを渡す
plot(dnorm, xlim = c(-3, 3))

curve(dnorm, from = -3, to = 3)

### (a-2) 標準正規分布を描画、式を渡す
curve(dnorm(x), from = -3, to = 3, col = "red")

### (a-3) 正規分布を描画、式を渡す
curve(dnorm(x, mean = 0, sd = 1), -3, 3,
      col = "blue", ylab = "確率密度")

### (a-4) 標準正規乱数のヒストグラムと確率密度曲線
y <- rnorm(200)          # 標準正規乱数 200個
hist(y,                   
     xlim = c(-3, 3), 
     ylim = c(0, 0.5),　
     freq = FALSE)　　　 # 縦軸を確率密度に指定

m <- mean(y); s <- sd(y)

curve(dnorm(x, mean = m, sd = s), 
      col = "red",　　# 正規分布による近似曲線
      add = TRUE)　　　

lines(density(y), 　　# カーネル密度推定による
      col = "blue")　 # 近似曲線　　　

### (a-5) 標準正規分布と95%信頼区間
curve(dnorm, from = -3, to = 3)

xx <- seq(qnorm(0.025), qnorm(0.975), by = 0.01)
yy <- dnorm(xx)
x <- c(qnorm(0.025), xx, qnorm(0.975))
y <- c(0, yy, 0)

polygon(x, y, col = "lightblue")　# 95% 部分を着色

abline(h = 0)　　　　　　　　　　 # 水平線の追加

## (b) カイ２乗分布(Chi-square Distribution)

### (b-1) 自由度を変えたカイ２乗分布
df_list  <- c(1, 2, 5, 10, 20)       # 自由度の指定
col_list <- rainbow(length(df_list)) # 色の指定

curve(dchisq(x, df = df_list[1], ncp = 0),
      from =0, to=20, 
      ylab = "確率密度",
      col = col_list[1])　# 1番目の自由度の分布　　　　　　　

for (i in 2:5){　　　　　 
  curve(dchisq(x, df = df_list[i], ncp = 0),
        col = col_list[i],
        add = TRUE)}      # 2番目以降の自由度の分布

legend("topright", title = "自由度",
       legend = paste("df =", df_list), 
       col = col_list, 
       lwd = 2, cex = 0.8)

## (b-2) カイ２乗分布の両側確率5%(片側2.5%)の範囲
x <- seq(from = 0, to = 30, by = 0.01)　　 # x座標
y <- dchisq(x, df = 8) # xに対するカイ２乗値(y座標)

plot(x, y, type = "l")　# 確率密度曲線

q <- qchisq(p = c(0.025, 0.975), df = 8)
print(q)   # q[1]：下側2.5%点、q[2]：上側2.5%点

polygon(x = c(0, x[x <= q[1]], q[1]),　
        y = c(0, y[x <= q[1]], 0), 
        col = "red")　　# 下側2.5% の範囲

polygon(x = c(q[2], x[x >= q[2]],max(x)),
        y = c(0, y[x >= q[2]], 0), 
        col = "red")    # 上側2.5%の範囲

abline(h = 0)           # 水平線の追加

## (c) t 分布(t distribution)
curve(dt(x, df = 2, ncp = 0),
      from = -4, to = 4, 
      ylim = c(0,0.4), 
      ylab = "確率密度", main = "t 分布",
      col = "black", 
      las = 1)

curve(dt(x, df = 5), col = "red", add = TRUE)

curve(dt(x, df = 20), col = "blue", add = TRUE)

legend(x = 1.2, y = 0.4,  # 位置を座標軸で指示
       legend = c("df = 2", "df = 5", "df = 20"),
       lty = 1,
       col = c("black", "red", "blue"),
       title ="自由度",
       cex = 0.6, title.cex = 0.7, bty = "n")

## (d) F 分布(F Distribution)
curve(expr = df(x, df1 = 5, df2 = 2),
      from = 0,
      to = 5,
      ylab = "確率密度",
      main = "F 分布(自由度 5, 2)",
      las = 1)

## (e) ２項分布(Binomial Distribution)
##     公正なサイコロを10回投げたとき
##     6 の目が出る回数とその確率
plot(x = 0:10, 
     y = dbinom(x = 0:10, size = 10, prob = 1/6),
     type = "h", 
     xlab = "10 回中、6 の目が出る回数", 
     ylab = "確率",
     ylim = c(0, 0.4),
     las = 1)

## (f) ２次関数(Quadratic Function)
### (f-1)
x <- seq(from = -10, to = 10, by = 0.5)
y <- x ^ 4 - 5 * x

plot(x, y, 
     type = "l",
     col = "red",
     las = 1)

### (f-2)
curve(expr = x ^ 4 - 5 * x,
      from = -10,
      to   = 10,
      col = "blue",
      las = 1)


# 以上 -----------------------------------------
